function [r,gini] = solve_model(ggamma, gov_g, gov_b, gov_tr, p, z)

% Parameters
rho   = 0.04; % discount rate (annual)
ddelta= 0.1; % depreciation
aalpha= 1/3;  % capital share 
Aprod = 0.6;  % A (productivity term)
x0    = 1;     % initial value of TFP
Y0    = 1;     % initial value of Y
eta   = 0.01;  % TFP growth
% Effective discount rate
rho_f = rho - (1-ggamma)*eta;

% Assets grid
I     = 1000;
amin  = 0;
amax  = 300;
a     = linspace(amin,amax,I)';
da    = (amax-amin)/(I-1);
aa    = [a,a,a,a];

% Iteration parameters
maxit  = 1000;
crit   = 10^(-6);
Delta  = 1000;
Ir     = 1000;
crit_S = 10^(-5);

% stochastic income process
nz = 4;
zz    = ones(I,1)*z;
pp = p; %save for future reference

% Calculate intensities: main diagonal idicates outflow from the states,
% the other numbers in a given row inflows to other states
q = eye(4)-diag(diag(pp));
pp(1:size(p)+1:end) = diag(q); % replace diagonal of pp with diagonal of q

% Calculate average productivity in stationary distribtion.
% This can be computed outside of the model
tolst    = 10^(-6); % tolerance
z_dist_old = ones(1,nz)/nz;
maxdiff = 10;
while (maxdiff>tolst)
    z_dist_new =  z_dist_old*p;
    maxdiff = max(abs(z_dist_new-z_dist_old));
    z_dist_old = z_dist_new;
end

z_ave = sum(z.*z_dist_new);

% Transition matrix
Aswitch = [-speye(I)*pp(1,1),speye(I)*pp(1,2),speye(I)*pp(1,3),speye(I)*pp(1,4);...
           speye(I)*pp(2,1),-speye(I)*pp(2,2),speye(I)*pp(2,3),speye(I)*pp(2,4);...
           speye(I)*pp(3,1),speye(I)*pp(3,2),-speye(I)*pp(3,3),speye(I)*pp(3,4);...
           speye(I)*pp(4,1),speye(I)*pp(4,2),speye(I)*pp(4,3),-speye(I)*pp(4,4)];

% Initial guess for interest rate
r_pretax = 0.5*(ggamma*eta + rho);

% initial ranges for r_pretax
rmin = 0;
rmax = rho_f;

% Placeholders for derivatives of the value function (frwd, backward, central)
dVf   = zeros(I,nz);
dVb   = zeros(I,nz);
c     = zeros(I,nz);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                 1. LOOP OVER r               %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for ir=1:Ir
iteration = ir;
r_r(ir)   =r_pretax;
rmin_r(ir)=rmin;
rmax_r(ir)=rmax;

% Capital demand from the firm FOC
KD(ir) = (aalpha*Aprod/(r_pretax + ddelta))^(1/(1-aalpha))*(x0/Y0)*z_ave;
% Wage from the firm FOC
w_pretax      = (1-aalpha)*Aprod*(KD(ir).^aalpha)*((x0/Y0)^(1-aalpha))*(z_ave^(-aalpha));
% Implied tax rate from Gov budget constraint in the steady state
tau    = (gov_g + gov_tr + gov_b*(r_pretax-eta)) / (1-ddelta*KD(ir)+r_pretax*gov_b);

r = (1-tau)*r_pretax;
% if r<eta
%     r=eta+0.001;
% end
w = (1-tau)*w_pretax;

CHECK_cpos = min(w*z(1) + (r-eta).*a);

if min(w*z(1) + (r-eta)*a(1)) < 0
    disp('CAREFUL: borrowing constraint too loose, c negative')
    disp(min(w*z(1) + (r-eta).*a))
    [M,Index] = min(w*z(1) + (r-eta).*a);
    disp(Index)
end

% value functions for the first iteration (natural initial guess: staying
% put, i.e. constant stream of utility forever (discounted at effective discount rate)
v0(:,1) = (w*z(1) + (r-eta).*a).^(1-ggamma)/(1-ggamma)/rho_f;
v0(:,2) = (w*z(2) + (r-eta).*a).^(1-ggamma)/(1-ggamma)/rho_f;
v0(:,3) = (w*z(3) + (r-eta).*a).^(1-ggamma)/(1-ggamma)/rho_f;
v0(:,4) = (w*z(4) + (r-eta).*a).^(1-ggamma)/(1-ggamma)/rho_f;

% if not the first iteration, use previous value functions to start with
if ir>1
v0 = V_r(:,:,ir-1);
end

v = v0;

%%%%%%%%%%%%%%%%%%%%%%%%%%
%   SOLVE HJB EQUATION   %
%%%%%%%%%%%%%%%%%%%%%%%%%%

for n=1:maxit
    V = v;
    V_n(:,:,n)=V;
    % forward difference
    dVf(1:I-1,:) = (V(2:I,:)-V(1:I-1,:))/da;
    dVf(I,:) = (w*z + (r).*amax).^(-ggamma); %will never be used, but impose state constraint a<=amax just in case
    if min(dVf)<0
        disp('Warning: Forward difference negative')
        return
    end 
    % backward difference
    dVb(2:I,:) = (V(2:I,:)-V(1:I-1,:))/da;
    dVb(1,:) = (w*z + (r-eta).*amin).^(-ggamma); %state constraint boundary condition
    if min(dVb)<0
        disp('Warning: Backward difference negative')
        return
    end 
    %consumption and savings with forward difference
    cf = dVf.^(-1/ggamma);
    ssf = w*zz + (r-eta).*aa - cf;
    %consumption and savings with backward difference
    cb = dVb.^(-1/ggamma);
    ssb = w*zz + (r-eta).*aa - cb;
    %consumption and derivative of value function at steady state
    c0 = w*zz + (r-eta).*aa;
    
    % dV_upwind makes a choice of forward or backward differences based on
    % the sign of the drift    
    If = ssf > 0; %positive drift --> forward difference
    Ib = ssb < 0; %negative drift --> backward difference
    I0 = (1-If-Ib); %at steady state
    
    c = cf.*If + cb.*Ib + c0.*I0;
    u = c.^(1-ggamma)/(1-ggamma);
    
  if c<0
      disp("C negative")
      
  end
    
    % CONSTRUCT TRANSITION MATRIX: transitions result from policy
    % functions and from the transitions matrix across states
    X = -min(ssb,0)/da;
    Y = -max(ssf,0)/da + min(ssb,0)/da;
    Z = max(ssf,0)/da;
    
    A1=spdiags(Y(:,1),0,I,I)+spdiags(X(2:I,1),-1,I,I)+spdiags([0;Z(1:I-1,1)],1,I,I);
    A2=spdiags(Y(:,2),0,I,I)+spdiags(X(2:I,2),-1,I,I)+spdiags([0;Z(1:I-1,2)],1,I,I);
    A3=spdiags(Y(:,3),0,I,I)+spdiags(X(2:I,3),-1,I,I)+spdiags([0;Z(1:I-1,3)],1,I,I);
    A4=spdiags(Y(:,4),0,I,I)+spdiags(X(2:I,4),-1,I,I)+spdiags([0;Z(1:I-1,4)],1,I,I);
   
    A = [A1,sparse(I,I),sparse(I,I),sparse(I,I);...
        sparse(I,I),A2,sparse(I,I),sparse(I,I);...
        sparse(I,I),sparse(I,I),A3,sparse(I,I);...
        sparse(I,I),sparse(I,I),sparse(I,I),A4] + Aswitch;
    
    if max(abs(sum(A,2)))>10^(-9)
       disp('Improper Transition Matrix')
         return
    end
    
    B = (1/Delta + rho_f)*speye(4*I) - A;

    u_stacked = [u(:,1);u(:,2);u(:,3);u(:,4)];
    V_stacked = [V(:,1);V(:,2);V(:,3);V(:,4)];
    
    b = u_stacked + V_stacked/Delta;
    
    % Solve the system of equations
    V_stacked = B\b; 
    
    % Unstack V
    V = [V_stacked(1:I),V_stacked(I+1:2*I),V_stacked(2*I+1:3*I),V_stacked(3*I+1:4*I)];
    
    % Calculate change between iterations
    Vchange = V - v;
    v = V;
    


    % Assess convergence
    dist(n) = max(max(abs(Vchange)));
    if dist(n)<crit
        %disp('Value Function Converged, Iteration = ')
        %disp(n)
        break
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%
% FOKKER-PLANCK EQUATION %
%%%%%%%%%%%%%%%%%%%%%%%%%%

AT = A';
b  = zeros(nz*I,1);

%need to fix one value, otherwise matrix is singular
i_fix       = 1;
b(i_fix)    =.1;
row         = [zeros(1,i_fix-1),1,zeros(1,nz*I-i_fix)];
AT(i_fix,:) = row;

%Solve linear system
gg    = AT\b;
g_sum = gg'*ones(nz*I,1)*da;
% Reweight - this is necessary because recall we solved with the fixed
% value at an arbitrary constant 0.1.
gg    = gg./g_sum;

% unstack the distribution
g     = [gg(1:I),gg(I+1:2*I),gg(2*I+1:3*I),gg(3*I+1:4*I)];

check1 = g(:,1)'*ones(I,1)*da;
check2 = g(:,2)'*ones(I,1)*da;
check3 = g(:,3)'*ones(I,1)*da;
check4 = g(:,4)'*ones(I,1)*da;

CHECK = 1 - check1 - check2 - check3 -check4;

% save results
g_r(:,:,ir)  = g;
adot(:,:,ir) = w*zz + (r-eta).*aa - c;
V_r(:,:,ir)  = V;

% calculate capital supply and disequilibrium
KS(ir) = g(:,1)'*a*da + g(:,2)'*a*da + g(:,3)'*a*da + g(:,4)'*a*da;
S(ir)  = KS(ir) - KD(ir) - gov_b;


% Calulcations to get % of people within $500 of borrowing constraint
First_bucket           = (da/KS(ir))*100; %how far away from the actual constraint are the constrained people, in percent of total assets
Constraint_index       = round(5/First_bucket); %how many buckets to take in to give fraction with less than 5% of assets
Fraction_at_constraint = sum(sum(g(1:Constraint_index,:)))/sum(sum(g(:,:))); %fraction of people at constraint defined as less than 5% of average assets holdings

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% UPDATE THE INTEREST RATE %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if S(ir)>crit_S
    %disp('Excess Supply')
    %disp(r_pretax)
    rmax = r_pretax;
    r_pretax = 0.5*(r_pretax+rmin);
    %disp(r_pretax)
elseif S(ir)<-crit_S;
    %disp('Excess Demand')
    %disp(r_pretax)
    rmin = r_pretax;
    r_pretax = 0.5*(r_pretax+rmax);
    %disp(r_pretax)
elseif abs(S(ir))<crit_S;
    %display('Equilibrium Found, Interest rate (pre-tax) =')
    %disp(r_pretax)
    %display('Interest rate (post-tax) =')
    %disp(r)
    %display('r - g =') 
    %disp(r-eta)
    %display('K =') 
    %disp(KD(ir))
    %display('Fraction of consumers at the constraint = ')
    %disp(Fraction_at_constraint)
    break
end

end
%income gini
    y = w*zz + r.*aa;
    Ny = nz*I;
    yy =reshape(y,Ny,1);
    [yy,index] = sort(yy);
    g_y = gg(index)*da;
    S_y = cumsum(g_y.*yy)/sum(g_y.*yy);
    trapez_y = (1/2)*(S_y(1)*g_y(1) + sum((S_y(2:Ny) + S_y(1:Ny-1)).*g_y(2:Ny)));
    gini = 1 - 2*trapez_y;


    

end 
